#----------------------------------------------------------------------------
# Copyright (c) 1998-2008 Autodesk, Inc.
# All rights reserved.
#
# These coded instructions, statements, and computer programs contain
# unpublished proprietary information written by Autodesk, Inc., and are
# protected by Federal copyright law. They may not be disclosed to third
# parties or copied or duplicated in any form, in whole or in part, without
# the prior written consent of Autodesk, Inc.
#----------------------------------------------------------------------------
# Author: Nicolas Desjardins
# Date: 2008-04-02
#----------------------------------------------------------------------------
# Driver script for fully automated unit test execution. 
# - Runs all dlls and plug-ins found in the build through the NUnit console  
# runner, executing any .NET unit tests found.
# - Runs all dlls and plug-ins found in the build through the CppUnit console
# runner, executing any C++ unit tests found.
# - Searches the source branch for any maxscripts named *unittest.ms.  Runs
# any it finds through the MxsUnit runner, grouped by directory.
#
# Run unit_test_everything.rb --help for usage information.
 
require 'rake'
require 'optparse'

class Runner
  def initialize
    parse_options()
    
    @build           = Dir.getwd unless @build # Get the current working directory
    @scripts         = determine_scripts_root    unless @scripts
    @nunit_command   = determine_nunit_command   unless @nunit_command
    @cppunit_command = determine_cppunit_command unless @cppunit_command
    @mxsunit_command = determine_mxsunit_command unless @mxsunit_command
    
    determine_dll_list()
    
    configure_environment()
    create_tasks()
  end
  
  def parse_options
    opts = OptionParser.new(nil, 32, '  ')
    opts.banner = "Usage: #{File.basename(__FILE__)} [options] [targets]\n\n" +
      "Targets:\n"+
      "  nunit    Run NUnit tests\n"+
      "  cppunit  Run CppUnit tests\n" +
      "  mxsunit  Run MxsUnit tests\n" +
      "  By default, if no targets are specified, runs all three.\n\n" +
      "Options:\n"
    
    rake_options = []
    @max_options = "-silent" 
    
    opts.on("--build=PATH", 
      "Path to max build directory.\n    Default is the current working directory.\n") do |value|
      @build = File.expand_path(value)
    end
    
    opts.on("--scripts=PATH", 
      "Path to directory in which to search\n    recursively for scripts. Default is 3dswin/src above the build directory.\n    This also sets the UNIT_TEST_DATA environment variable.") do |value|
      @scripts = File.expand_path(value)
      ENV['UNIT_TEST_DATA'] = @scripts
    end
    
    opts.on("-yc", "Run max with network licensing.\n    Cannot be used with --max_options.\n") do |value|
      @max_options += " -yc"
    end
    
    opts.on("--max_options=\"LIST\"", 
      "Options to pass to max.\n    For just network licensing, it's simpler to use -yc.\n") do |value|
      @max_options = value
    end
    
    opts.on("--nunit_command=\"CMDLINE\"", 
      "Path to NUnit runner.\n    By default, attempts to find the runner in the build directory.\n") do |value|
      @nunit_command = value
    end
    
    opts.on("--cppunit_command=\"CMDLINE\"", 
      "Path to CppUnit runner.\n    By default, attempts to find the runner in the build directory.\n") do |value|
      @cppunit_command = value
    end
    
    opts.on("--mxsunit_command=\"CMDLINE\"", 
      "Command line used to run MxsUnit tests.\n    By default, attempts to find runner, ProcessHandler\n    and 3dsmax or 3dsviz in the build directory.\n") do |value|
      @mxsunit_command = value
    end
    
    opts.on("-n", "--dry-run", "Do a dry run without executing tests\n") do |value|
      rake_options << "--dry-run"
    end
    
    opts.on("-h", "-?", "--help", "Print usage information.\n") do |value|
      puts opts.to_s
      exit
    end
    
    remaining = opts.parse(ARGV)
    ARGV.clear
    ARGV.push(*rake_options) unless rake_options.empty?
    ARGV.push(*remaining)
  end
  
  def determine_scripts_root
    # the root of all maxscripts for mxsunit defaults to the 3dswin/src in a 
    # standard max stage
    full_build_root = File.expand_path(@build)
    matchdata = full_build_root.match(/(^.*\/src\/)/)
    if matchdata
      matchdata[1]
    else
      fail "Cannot determine root search directory for mxsunit scripts, use -s -scripts=<path>"        
    end
  end
    
  def determine_nunit_command
    x86_command = File.join(@build, "nunit-console-x86.exe")
    plain_command = File.join(@build, "nunit-console.exe")
    if File.exists? x86_command
      "#{x86_command} /nologo"
    elsif File.exists? plain_command
      "#{plain_command} /nologo"
    else
      fail "Unable to find NUnit runner: #{x86_command} or #{plain_command}, use -nunit_command=<path>"
    end
  end
  
  def determine_cppunit_command
    cppunit_runner = File.join(@build, "CppUnitRunner.exe")
    if File.exists? cppunit_runner
      cppunit_runner
    else
      fail "Unable to find CppUnit runner: #{cppunit_runner}, use -cppunit_command=<path>"
    end
  end
  
  def determine_mxsunit_command
    mxsunit_runner = File.join(@build, "mxsunit.rb")
    unless File.exists? mxsunit_runner
      fail "Unable to find MxsUnit runner: #{mxsunit_runner}, use -mxsunit_command=\"<mxsunit command line>\""
    end
    
    process_harness_command = File.join(@build, "ProcessHarness.exe")
    unless File.exists? process_harness_command
      fail "Unable to find ProcessHarness utility: #{process_harness_command}, use -mxsunit_command=\"<mxsunit command line>\""
    end
    
    max = File.join(@build, "3dsmax.exe")
    viz = File.join(@build, "3dsviz.exe")
    max_command = 
      if File.exists? max
        max
      elsif File.exists? viz
        viz
      else
        fail "Unable to find max executable: #{max} or #{viz}, use -mxsunit_command=\"<mxsunit command line>\""
      end
      
    %Q{ruby "#{mxsunit_runner}" "#{process_harness_command}" "#{max_command}" #{@max_options}}
  end
  
def determine_dll_list
	build_test_list = File.join(@build, "build_test_list.txt")
	
	@csproj_list = Array.new
	@vcxproj_list= Array.new
	
	if File.exists? build_test_list
		puts "Found build_test_list.txt"
		# This is a fast way to unit test only the files that are actually 
		# built by the max build.
		File.open(build_test_list,"r").each_line do |line|
			data = line.split(/-/)
			if (data[0] == ".csproj")
				@csproj_list.push(data[1])
			elsif (data[0] == ".vcxproj")
				@vcxproj_list.push(data[1])
			end
		end
		puts "Found " + @csproj_list.length.to_s + " .csproj files"
		puts "Found " + @vcxproj_list.length.to_s + " .vcxproj files"
	else
		puts "Did not find build_test_list.txt. Build an inefficient list of files to test."
		puts "This will take a long time to run."
		@dll_list = FileList[File.join(@build, '**/*.{exe,dll,dlz,dlu,dlo,dlm,dlc,dli,dlt,dle,dlv,dlk,dlb,dlf,dlr,bmf,bmi,bms,flt,gup,dlh,dln,dla,dls,dly}')]
		@dll_list.exclude(File.join(@build, "nunit*"))
		@dll_list.exclude(File.join(@build, "cppunit_dll.dll"))
	end
end

def configure_environment
	# set up the environment so DLLs off the main directory can find their 
	# dependencies.
	build_path          = windows_slashes(File.expand_path(@build))
	bin_assemblies_path = windows_slashes(File.join(build_path, "bin\\assemblies"))
	
	ENV['PATH'] += ";" + build_path + ";" + bin_assemblies_path
end

def create_tasks
	task :default => [ :nunit, :cppunit, :mxsunit ]

	task :nunit do
		@nunit_ran = true
		@assemblies = 0
		@nunit_tested = 0
		@nunit_failed = 0
		
		temp_list = @csproj_list
		if (@csproj_list.length == 0)
			temp_list = @dll_list
		end
		
		temp_list.each do |f|
			command = "#{@nunit_command} #{windows_slashes f} 2>&1"
			puts "#{@nunit_command} #{windows_slashes f}"
			result = `#{command}`
			if result.match "Tests run"
				puts result
				@assemblies += 1
				@nunit_tested += 1 if result.match(/Tests run: [^0]/)
				@nunit_failed += 1 if result.match(/Failures: [^0]/)
			end
		end
	end

	task :cppunit do
		@cppunit_ran = true
		@cppunit_tested = 0
		@cppunit_failed = 0
		
		temp_list = @vcxproj_list
		if (@vcxproj_list.length == 0)
			temp_list = @dll_list
		end
		
		temp_list.each do |f|
			command = "#{@cppunit_command} #{windows_slashes f} 2>&1"
			puts "#{@cppunit_command} #{windows_slashes f}"
			result = `#{command}`
			puts result
			# Note that we ignore DLLs that fail to load and consider them untested
			# Too many DLLs without tests fail due to link dependency search order 
			# weirdness.
			#
			# We try our best here and within the runner, but it's up to the 
			# programmer responsible for that test to make sure that the test runs.
			if result.match(/OK \([^0]|FAILURES/)
				@cppunit_tested += 1
				@cppunit_failed += 1 unless result.match(/OK \([^0]/)
			elsif 0 < $?.exitstatus 
				# Some errors cause the runner to crash in very evil ways.  They set
				# the exit code to a value greater than 1.  We'll find these too. 
				puts "Error: Runner crashed with exit code #{$?.exitstatus}"
				puts
				@cppunit_tested += 1
				@cppunit_failed += 1
			end
		end
	end

	task :mxsunit do
		@mxsunit_ran = true
		# run scripts grouped together by directory
		# This seems to be a reasonable compromise between how many times we open
		# and close max and preventing maxscript errors and crashes in some tests
		# from interfering with other tests.
		@mxsunit_tested = 0
		@mxsunit_failed = 0
		FileList[ File.join(@scripts, '**/') ].each do |dir|
			script_list = FileList[ File.join(dir, '*unittest.ms') ]
			unless script_list.empty?
				script_list.collect! { |t| %Q{"#{t}"} }
				command = "#{@mxsunit_command} -t #{script_list.join ' '}"
				puts command
				result = `#{command}`
				exit_code = $?.exitstatus
				puts result
				@mxsunit_tested += 1
				unless ( result.match(/OK \(/) && 
					result.match(/MxsUnit completed/) && 
					0 == exit_code )
					@mxsunit_failed += 1 
				end
			end
		end
	end
end
  
def print_results_summary
	puts
	if @nunit_ran
		puts "NUnit Results"
		if (@csproj_list.length > 0)
			puts "  .NET Assemblies: #{@csproj_list.length}, Tested: #{@nunit_tested}, Failed #{@nunit_failed}"
		else
			puts "  Binaries: #{dll_list.length}, .NET Assemblies: #{@assemblies}, Tested: #{@nunit_tested}, Failed #{@nunit_failed}"
		end
	end

	if @cppunit_ran
		puts "CppUnit Results"
		if (@vcxproj_list.length > 0)
			puts "  Binaries: #{@vcxproj_list.length}, Tested: #{@cppunit_tested}, Failed: #{@cppunit_failed}"
		else
			puts "  Binaries: #{dll_list.length}, Tested: #{@cppunit_tested}, Failed: #{@cppunit_failed}"
		end
	end

	if @mxsunit_ran
		puts "MxsUnit Results"
		puts "  Script suites ran: #{@mxsunit_tested}, Failed: #{@mxsunit_failed}"
	end

	if all_ok?
		puts "OK"
	else
		puts "Failed"
	end
end
  
def all_ok?
	all_ok = true
	if @nunit_ran
		all_ok = all_ok && (0 == @nunit_failed)
	end

	if @cppunit_ran
		all_ok = all_ok && (0 == @cppunit_failed)
	end

	if @mxsunit_ran
		all_ok = all_ok && (0 == @mxsunit_failed)
	end
	all_ok
end
  
def run
	Rake.application.options.trace = true
	Rake.application.run
	print_results_summary
	all_ok?
end

def windows_slashes path
	path.gsub(/\//, "\\")
end
  
end

module Rake
  class Application
    def load_rakefile
      # never look for a rakefile, tasks are defined in the class above
    end
    
    def have_rakefile
      true
    end
  end
end


result = Runner.new.run
# use an exit code of 1 to signal a failure
exit(1) unless result

